/*
 *  bootload.c      68hc11 bootloader, using COM: port from PC.
 */

#include  <stdio.h>
#include  <bios.h>
#include  <string.h>


void  ProcessRecord();
void  ShowError(int  errnum);
void  DownloadFile();


#define  progname   "bootload"
#define  version    "1.0"

#define  ESC    0x1b
#define  CPU_A1    0x01
#define  CPU_E1    0x02
#define  CPU_F1    0x03

#define  ERR_NUM  13                 /* number of error messages */

#define  ERR_NOFILE        0
#define  ERR_BADREC        1
#define  ERR_NOSUPPORT     2
#define  ERR_BADADDR       3
#define  ERR_BADCHKSUM     4
#define  ERR_XMITTO        5
#define  ERR_RCVTO         6
#define  ERR_ECHO          7
#define  ERR_BADPORT	   8
#define  ERR_BADPORTARG	   9
#define  ERR_BADFILENAME  10
#define  ERR_BADSERIAL    11
#define  ERR_BADCPU       12



char  *errtxt[ERR_NUM+1] = {
	"Unable to open specified .S19 bootstrap file.",        /* nofile */
	"Bad S19 record.",                                      /* bad record */
	"Unsupported S-record format; must be S0, S1 or S9.",   /* bad format */
	"Address is out of range for this MCU.",                /* bad addr */
	"Record checksum is bad.",                              /* bad checksum */
	"COM port timed out on transmit.",						/* xmit time out */
	"COM port timed out on receive.",						/* rcv time out */
	"68hc11 failed to properly echo a transmitted byte.",	/* echo failure */
	"Unsupported COM: port number, must be 1 or 2.",		/* bad port */
	"Illegal /p argument, must be of form /pn.",			/* bad port arg */
	"Illegal character in file name.",						/* bad file name */
	"Timeout on COM: port; 68hc11 did not respond.",		/* bad serial */
	"Unknown CPU type; must be A1 or E1.",					/* bad cpu */
	""};

unsigned char   ram[1024];          /* loading area for S19 bootstrap code */
unsigned int	max_addr;			/* highest legal addr for this cpu */
int             port;               /* holds COM port number (0 = COM1:) */
char            filename[80];       /* holds string for bootstrap file name */
int             linenum;            /* tracks line number in bootstrap file */
char            srec[256];          /* holds S-record from bootstrap file */
unsigned int    status;             /* general status variable */
FILE            *bootptr;           /* pointer to bootstrap file */
int             cputype;            /* holds type of cpu involved */
int             n, j;               /* temp registers */


void  main(int  argc, char  *argv[], char *envp[])
{
	printf("\n%s version %s", progname, version);
	
	if (argc == 1)  {
		printf("\nUsage:  %s  bootfile  <options>"
			   "\n  where bootfile is the path to a .S19 bootstrap file and"
			   "\n  <options> are zero or more from the following list..."
			   "\n  /pn     set port number; n=1 for COM1:, etc."
			   "\n  /ma1    use 68hc11a0, a1, or a8 CPU (download 256 bytes)."
			   "\n  /me1    use 68hc11e1, e2, or e9 CPU (download 512 bytes)."
			   "\n  /mf1    use 68hc11f1 (download 1024 bytes)."
			   "\n  "
			   , progname);
		exit(1);
	}


/*
 *  Set up the system defaults before processing any arguments.
 */

	port = 0;									/* default COM: port */
	cputype = CPU_A1;							/* default CPU type */

	strcpy(filename,argv[1]);					/* get the file name */
	if (strchr(filename, '/') != NULL)  {
		ShowError(ERR_BADFILENAME);
		exit(2);
	}

	if ((strstr(filename, ".s19") == NULL) &&
		 (strstr(filename, ".S19") == NULL))  {
		 	strcat(filename, ".S19");
	}
	
	for (n=2; n<argc; n++)  {
		if (strstr(argv[n], "/p") == argv[n])  {
			if (strlen(argv[n]) == 3)  {
				port = (*(argv[n]+2) - '0') - 1;
				if ((port != 0) && (port != 1))  {
					ShowError(ERR_BADPORT);
					exit(2);
				}
			}
			else  {
				ShowError(ERR_BADPORTARG);
				exit(2);
			}
		}
		if (strstr(argv[n], "/m") == argv[n])  {
			if (strlen(argv[n]) == 4)  {
				if ((strstr(argv[n], "a1") != NULL)  ||
					(strstr(argv[n], "A1") != NULL))  {
						cputype = CPU_A1;
				}
				else if ((strstr(argv[n], "e1") != NULL) ||
						(strstr(argv[n], "E1") != NULL))  {
						cputype = CPU_E1;
				}
				else if ((strstr(argv[n], "f1") != NULL) ||
						(strstr(argv[n], "F1") != NULL))  {
						cputype = CPU_F1;
				}
				else  {
					ShowError(ERR_BADCPU);
					exit(2);
				}
			}
			else  {
				ShowError(ERR_BADCPU);
				exit(2);
			}
		}
	}
	
	linenum = 0;
	for (n=0; n<512; n++)  ram[n] = 0;			/* clear the RAM buffer */

	bootptr = fopen(filename, "r");
	if (!bootptr)  {
		ShowError(ERR_NOFILE);
		exit(1);
	}

	if (cputype == CPU_A1)  max_addr = 0xff;
	if (cputype == CPU_E1)  max_addr = 0x1ff;
	if (cputype == CPU_F1)  max_addr = 0x3ff;

	while (!feof(bootptr))  {
		linenum++;
		fgets(srec, 256, bootptr);
		ProcessRecord();
	}

	fclose(bootptr);
	linenum = 0;

	DownloadFile();
}



void  ProcessRecord()
{
	int     cnt;
	int     chksum;
	int     tchksum;
	int     sdata;
	int     addr;
	int     n;

	if (srec[0] == '\0')  return;           /* just in case */
	if (srec[0] == '\n')  return;           /* just in case */
	if (srec[0] == '*')  return;            /* * in column 1 = comment */
	if (srec[0] != 'S')  {                  /* no S in column 1... */
		ShowError(ERR_BADREC);              /* show bad record format */
		exit(2);                            /* exit fatally */
	}
	if (srec[1] == '9')  return;            /* S9 = EOF, ignore it */
	if (srec[1] == '0')  return;            /* S0 = header, ignore it */
	if (srec[1] != '1')  {                  /* not S1, unsupported record */
		ShowError(ERR_NOSUPPORT);           /* show the error */
		exit(2);
	}

	sscanf(srec+2, "%2x", &cnt);            /* get the number of bytes in rec */
	chksum = cnt;
	sscanf(srec+4, "%4x", &addr);           /* get addr of this rec */
	chksum += (addr >> 8);
	chksum += (addr & 0xff);

	for (n=2; n<(cnt-1); n++)  {
		sscanf(srec+4+(n*2), "%2x", &sdata);
		chksum += sdata;
		if (addr > max_addr)  {
			ShowError(ERR_BADADDR);
			exit(2);
		}
		ram[addr] = sdata;
		addr++;
	}
	sscanf(srec+2+(cnt*2), "%2x", &tchksum);
	if ((tchksum + (chksum & 0xff)) != 0xff)  {
		ShowError(ERR_BADCHKSUM);
		exit(2);
	}
}


void  ShowError(int  errnum)
{
	printf("\nError: %s", errtxt[errnum]);
	if (linenum)  {
		printf("\n%s, line %d:  %s", filename, linenum, srec);
	}
}



void  DownloadFile()
{
	int		c;

	printf("\n\nReady to download file %s (%d bytes) using COM%d.",
		 filename, max_addr+1, port+1);
	printf("\n\nPut target 68hc11 system in special bootstrap mode, reset");
	printf("\nthe chip, then press RETURN to download or ESC to quit: ");

	do  {
		c = getch();
		if (c == ESC)  {
			printf("\n\nDownload aborted by operator.");
			exit(2);
		}
	}  while (c != '\r');

	printf("\n\nDownload started...");

	status = (_bios_serialcom(_COM_INIT, port,
				_COM_1200 | _COM_CHR8 | _COM_NOPARITY | _COM_STOP1));
				
	status = _bios_serialcom(_COM_SEND, port, 0xff);
	if (status & 0x8000)  {
		ShowError(ERR_BADSERIAL);
		exit(2);
	}
	status = _bios_serialcom(_COM_RECEIVE, port, 0);
	if (status & 0x8000)  {
		ShowError(ERR_BADSERIAL);
		exit(2);
	}

	for (n=0; n<=max_addr; n++)  {
		status = _bios_serialcom(_COM_SEND, port, ram[n]);
		if (status & 0x8000)  {					/* if xmit timed out... */
			ShowError(ERR_XMITTO);				/* show the error */
			exit(2);
		}
		status = _bios_serialcom(_COM_RECEIVE, port, 0);
		if ((status & 0xff) != ram[n])  {		/* if handshake echo failed... */
			ShowError(ERR_ECHO);				/* show the error */
			exit(2);
		}
	}

	printf("\n\nFile %s successfully downloaded.", filename);
	exit(0);
}
