/*
 *  x10dos.c      Z-10 controller for DOS
 *
 *  This program runs from the DOS prompt.  When invoked, it opens the
 *  selected COM port, sends a stream of X-10 commands to that port, then
 *  shuts off the port.
 *
 *  Invocation is:
 *
 *  x10dos  <port>  <x10 commands>
 *
 *  where <port> is required and may be either 1 or 2 for COM1 or COM2.  If
 *  the <x10 commands> field is present, the arguments are interpreted as
 *  x10 commands and issued to the selected COM port.
 */

#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <dos.h>
#include  <ctype.h>
#include  <time.h>
#include  "fasttimr.h"


#ifndef  FALSE
#define  FALSE  0
#define  TRUE  !FALSE
#endif


/*
 *  Declare a set of macros for manipulating the RTS and DTR lines
 *  of the selected port.  Note that these macros use the variable
 *  port, which will be modified to hold the I/O address of the
 *  the selected COM port.
 */

#define  MCR  4
#define  RTS_ON  (outportb(port+MCR, (inportb(port+MCR) | 0x02)))
#define  RTS_OFF (outportb(port+MCR, (inportb(port+MCR) & 0xfd)))
#define  DTR_ON  (outportb(port+MCR, (inportb(port+MCR) | 0x01)))
#define  DTR_OFF (outportb(port+MCR, (inportb(port+MCR) & 0xfe)))

/*
 *  Define the program name and version as strings.
 */

#define  PROGNAME   "x10dos"
#define  VERSION    "1.3"
#define  HOUSEDEFS  "house.def"


/*
 *  Define some maximums.
 */

#define  MAXHOUSEDEFS  15
#define  MAXNAMELEN    40
#define  MAXUNITSLEN   100
#define  MAXSTATELEN   10




/*
 *  Define some magic commands for the cm17a.
 */

#define HEADER 		0xd5aa
#define FOOTER 		0xad
#define ON     		0x0000
#define OFF    		0x0020
#define BRIGHT005	0x0088
#define DIM005		0x0098
#define INVALIDCOMMAND 0xffff
#define  REPEAT_COUNT	2


const unsigned int HouseCodes[] =
{
  0x6000, /* a */
  0x7000, /* b */
  0x4000, /* c */
  0x5000, /* d */
  0x8000, /* e */
  0x9000, /* f */
  0xa000, /* g */
  0xb000, /* h */
  0xe000, /* i */
  0xf000, /* j */
  0xc000, /* k */
  0xd000, /* l */
  0x0000, /* m */
  0x1000, /* n */
  0x2000, /* o */
  0x3000  /* p */
} ;

const int MAXHOUSECODE = sizeof(HouseCodes) / sizeof(HouseCodes[0]);

const unsigned int UnitCodes[] =
{
  0x0000, /* 1 */
  0x0010, /* 2 */
  0x0008, /* 3 */
  0x0018, /* 4 */
  0x0040, /* 5 */
  0x0050, /* 6 */
  0x0048, /* 7 */
  0x0058, /* 8 */
  0x0400, /* 9 */
  0x0410, /* 10 */
  0x0408, /* 11 */
  0x0400, /* 12 */
  0x0440, /* 13 */
  0x0450, /* 14 */
  0x0448, /* 15 */
  0x0458  /* 16 */
} ;

const int MAXUNITCODE = sizeof(UnitCodes) / sizeof(UnitCodes[0]);

typedef  struct  defs_t
		{
			char		name[MAXNAMELEN];
			char		units[MAXUNITSLEN];
		}  DEFS;


unsigned int far		*ptr;
unsigned int			port;
unsigned char			portc;
unsigned char			oldmcr;
int						done;
char					buf[100];
char					state[MAXSTATELEN];
char					units[MAXUNITSLEN];
char					housefn[100];
FILE					*housef;
int						numdefs;
int						lineknt;

DEFS					housedefs[MAXHOUSEDEFS];




/*
 *  Define delays used in X10 communications.  All delay
 *  values are in usecs.
 */

#define  DLY_BIT		1000L
#define  DLY_RESET  	1000000L
#define  DLY_COMMAND	5000L


/*
 *  Local function definitions
 */

void  			Delay(unsigned long int  usecs);
void  			DoCommand(char  *units, char  *state);
void  			ReadHouseDefs(FILE  *hf);
void  			AddDef(char  *name, char  *def);
int  			LookupName(char  *name);
void 			Cm17aOut(unsigned char  b);
void 			Cm17aOutW(unsigned int  w);
void  			Cm17aReset();



int  main(int  argc, char *argv[])
{
	int					n;
	struct FASTCLOK		t;

	printf("\n%s version %s", PROGNAME, VERSION);

	if (argc == 1)  {
		printf("\nUsage: %s  <port>  <commands>", PROGNAME);
		printf("\nwhere <port> is 1 for COM1 or 2 for COM2 and");
		printf("\n<commands> is one or more x10 commands separated by spaces.");
		printf("\nLegal x10 commands are strings of the form:  hu cmd");
		printf("\nwhere h is the house code (a-f), u is the unit code (0-15),");
		printf("\nand cmd is a command string such as on, off, dim, or bright.");
		printf("\n");
		exit(0);
	}
	if ((strlen(argv[1]) != 1) || (strchr("12", *argv[1]) == 0))  {
		printf("\nError: Bad COM port argument; must be 1 or 2!");
		exit(1);
	}

	strcpy(housefn, HOUSEDEFS);
	housef = fopen(housefn, "r");
	if (housef)  {								// if able to open defs file...
		printf("\nUsing house definitions file %s.", housefn);
		ReadHouseDefs(housef);
	}
	ptr = (unsigned int far *)0x0400;
	portc = *argv[1];
	if (portc == '1')  	port = *ptr;
	else   				port = *(ptr + 1);
	if (port == 0)  {
		printf("\nError: COM%d (at %0x) not available.", portc, port);
		exit(1);
	}
	oldmcr = inportb(port+MCR);

	start_fastclock();

	done = FALSE;								// not done yet
	for (n=2; n<argc; )  {						// for all commands...
		strcpy(units, argv[n]);					// get the unit involved
		n++;									// move to next token
		if (n == argc)  {						// if no token...
			printf("\nError: Missing ON/OFF argument!.");
			exit(1);
		}
		strncpy(state, argv[n], MAXSTATELEN-1);		// copy requested state
        state[MAXSTATELEN-1] = '\0';			// force an end, just in case
		DoCommand(units, state);				// see what's going on
		if (done)  break;						// leave early if done
		n++;									// move to next token
	}
	printf("\n");
	outportb(port+MCR, oldmcr);					// restore modem lines
	return 0;
}



void  DoCommand(char  *units, char  *state)
{
	unsigned int				cmd;
	int							house;
	int							u;
	unsigned char				uknt;
	char						*p;
	char						*pg;
	char						*punit;
	int							n;
	int							ndef;
	char						tempunit[MAXUNITSLEN];
	time_t 						t;

//	printf("\n0: %s", units);
	strcpy(tempunit, units);
	ndef = LookupName(units);
	if (ndef != -1)  strcpy(tempunit, housedefs[ndef].units);
	punit = strtok(tempunit, " "); 		// point to first name in def
//	printf("\n    1: %s", punit);
	p = punit;							// set the working pointer
	ndef = LookupName(punit);			// try to find the name
	if (ndef != -1)  p = housedefs[ndef].units;	// use new definition

	do  {
//		printf("\n        2: %s", p);
		do  {
			while (*p == ' ')  p++;			// skip any whitespace
			cmd = 0;
			house = (int)toupper(*p) - 'A';
			if ((house < MAXHOUSECODE) && (house >= 0))  {
				cmd |= HouseCodes[house];
				time(&t);
				printf("\nAt %s, ", ctime(&t));
				printf("house %c, ", house + 'A');
				p++;
				u = 0;
				uknt = 0;
				if (*p == '*')  {				// if all units...
					printf("all units, ");      // show it
				}
				else  {							// try for unit number
					while (isdigit(*p))  {
						u = (u * 10) + (*p - '0');
						p++;
						uknt++;
					}
					if (uknt)  {
						printf("unit %d, ", u);
					}
					else  {
						printf("\nUnknown unit name!");
						return;
					}
				}

				if ((u < MAXUNITCODE) && (u > 0))  {
					u = UnitCodes[u-1];
				}
				else  {
					cmd = INVALIDCOMMAND;
				}

				if (strlen(state) == 0)  {
					printf(" ON");
					cmd |= (ON | u);
				}
				else  if (stricmp(state, "ON") == 0)  {
					printf(" ON");
					cmd |= (ON | u);
				}
				else  if (stricmp(state, "OFF") == 0)  {
					printf(" OFF");
					cmd |= (OFF | u);
				}
				else  if (stricmp(state, "DIM") == 0)  {
					printf(" DIM");
					cmd |= DIM005;
				}
				else if (stricmp(state, "BRIGHT") == 0)  {
					printf(" BRIGHT");
					cmd |= BRIGHT005;
				}
				else  {
					printf(" UKNOWN!");
					cmd = INVALIDCOMMAND;
				}
			}
			else  {
				cmd = INVALIDCOMMAND;
			}

			if (cmd != INVALIDCOMMAND)  {
				Cm17aReset();
				for (n=0; n<REPEAT_COUNT; n++)  {
					Delay(DLY_COMMAND);
					Cm17aOutW(HEADER);
					Cm17aOutW(cmd);
					Cm17aOut(FOOTER);
				}
			}
			if (*p)  p++;				 	// if more than one token, advance
		}  while (*p);					 	// do all units in this group
		punit = strtok(NULL, " ");			// point to next name in def
//		printf("\n    1: %s", punit);
		p = punit;							// set the working pointer
		ndef = LookupName(punit);			// try to find the name
		if (ndef != -1)  p = housedefs[ndef].units;	// use new definition
	}  while (p);
}





void  ReadHouseDefs(FILE  *hf)
{
	int					n;
	char				rbuf[MAXUNITSLEN];
	char				tname[MAXUNITSLEN];
	char				*p;
	int					r;

	numdefs = 0;							// start at beginning
	lineknt = 0;							// show line 0
	if (!hf)  return;						// don't waste time
	do  {
		do  {
			fgets(rbuf, MAXUNITSLEN-1, hf);			// read a line
			lineknt++;						// count it
		}  while (((*rbuf == '*') || (*rbuf == '\n')) && !feof(hf));
		if (feof(hf))  return;
		p = strchr(rbuf, '\n');				// find newline, if any
		if (p)  *p = '\0';					// replace with null
		p = strtok(rbuf, " ");				// get name of def
		if (strlen(p))  {					// if found a def...
			strcpy(tname, p);				// make a copy
			p = p + strlen(p) + 1;			// move past first token
            while (*p == ' ')  p++;			// move past leading spaces
			AddDef(tname, p);				// add definition to list
		}
	}  while (numdefs != MAXHOUSEDEFS);
}



void  AddDef(char  *name, char  *def)
{
	int					n;

	n = LookupName(name);
	if (n != -1)  {							// if already exists...
		printf("\nError: Duplicate definition %s at line %d.",
				name, lineknt);
		exit(1);
	}
	strcpy(housedefs[numdefs].name, name);
	strcpy(housedefs[numdefs].units, def);
//    printf("\nAdded %d: %s  %s", numdefs, name, def);
	numdefs++;
}



int  LookupName(char  *name)
{
	int				n;

	for (n=0; n<numdefs; n++)  {
		if (stricmp(name, housedefs[n].name) == 0)  {
			return  n;
		}
	}
	return  -1;
}






/*
 *  Cm17aReset()      reset the cm17a transmitter
 */

void  Cm17aReset()
{
	DTR_OFF;
	RTS_OFF;
	Delay(DLY_RESET);
	DTR_ON;
	RTS_ON;
	Delay(DLY_RESET);
	Delay(DLY_RESET);
}



/*
 *  Bit-bang the modem control lines to send a byte of data
 *  out the serial port to the cm17a.
 */

void Cm17aOut(unsigned char  b)
{
	int						i;

	i = 0x80;
	while (i)  {
		if (b & i) {
			RTS_ON;
			DTR_OFF;
		}
		else  {
			DTR_ON;
			RTS_OFF;
		}
		Delay(DLY_BIT);
		DTR_ON;
		RTS_ON;
		Delay(DLY_BIT);
		i >>= 1;
	}
}



/*
 *  Bit-bang the modem control lines to send a word of data
 *  out the serial port to the cm17a.
 */

void Cm17aOutW(unsigned int  w)
{
	unsigned int			i;

	i = 0x8000;
	while (i)  {
		if (w & i) {
			RTS_ON;
			DTR_OFF;
		}
		else  {
			DTR_ON;
			RTS_OFF;
		}
		Delay(DLY_BIT);
		DTR_ON;
		RTS_ON;
		Delay(DLY_BIT);
		i >>= 1;
	}
}


void  Delay(unsigned long int  usecs)
{
	int						n;
	struct  FASTCLOK		t;

	TimerSet(0, (time_t)0, usecs);
	while (!TimerZero(0))  ;
}




